﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using VA.PPMS.Context;

namespace VA.PPMS.CRM.Plugins
{
    public class HistoryLogValidationCreate : IPlugin
    {
        ITracingService _tracingService;
        IOrganizationService _service;
        private const string PluginName = "HistoryLogValidationCreate";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                Log("Begin");

                // Obtain the target entity from the input parameters.
                Entity target = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (target.LogicalName != "ppms_historylog")
                    return;

                Log("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                _service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    OptionSetValue historyLogType = target.GetAttributeValue<OptionSetValue>("ppms_type");

                    //If we dont have a History Log Type, end logic. 
                    if (historyLogType == null)
                        return;

                    Log("History Log Type: {0}", historyLogType.Value);

                    //If this is not a Validation History Log Type we don't want to run any of the logic.
                    if (historyLogType.Value != (int)PpmsHelper.HistoryLogType.Contact &&
                        historyLogType.Value != (int)PpmsHelper.HistoryLogType.Leie &&
                        historyLogType.Value != (int)PpmsHelper.HistoryLogType.NppesNpi &&
                        historyLogType.Value != (int)PpmsHelper.HistoryLogType.Sam &&
                        historyLogType.Value != (int)PpmsHelper.HistoryLogType.CareSite)
                    {
                        Log("Not a Validation History Log");
                        return;
                    }

                    // Get related entities to deactivate
                    Log("Retrieve provider related entities");
                    var providerRef = target.GetAttributeValue<EntityReference>("ppms_providerid");
                    Entity provider = GetProvider(_service, providerRef.Id);
                    if (provider == null)
                    {
                        Log("Failed to retrieve provider");
                        return;
                    }

                    List<SetStateRequest> requests = new List<SetStateRequest>();

                    // handle event based on message type
                    Log("Check History Log Validations");
                    var isComplete = IsValidationComplete(provider);

                    if (isComplete)
                    {
                        requests.AddRange(MarkBatchDetailAsComplete(provider));

                        // Determine if the provider can be activated
                        var currentStatus = provider.GetAttributeValue<OptionSetValue>("statecode");

                        if (currentStatus != null && currentStatus.Value == (int)AccountState.Active)
                        {
                            //If we get here we have the Provider and Validations are complete. 
                            Log("Set Provider to Active");
                            //Set Provider State/Status
                            requests.Add(PpmsHelper.GetStateRequest(providerRef, currentStatus.Value, (int)Account_StatusCode.Active));
                        }

                        // Execute requests
                        if (requests.Count > 0)
                        {
                            foreach (var request in requests)
                            {
                                _service.Execute(request);
                            }
                        }

                        return;
                    }
                    Log("Validations Not Complete");
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    Log("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    Log("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            Log("Done");
        }

        private Entity GetProvider(IOrganizationService service, Guid providerId)
        {
            var request = new RetrieveRequest();

            request.Target = new EntityReference("account", providerId);
            request.ColumnSet = new ColumnSet(new string[] { "accountid", "name", "statecode", "ppms_recordsource", "ppms_providertype" });
            request.RelatedEntitiesQuery = new RelationshipQueryCollection();

            //Retrieve related entities
            request.RelatedEntitiesQuery.Add(new Relationship("ppms_account_batchdetail_provider"),
                new QueryExpression("ppms_batchdetail")
                {
                    ColumnSet = new ColumnSet("ppms_batchdetailid", "ppms_name", "ppms_batch")
                }
            );

            //Get response
            var response = (RetrieveResponse)service.Execute(request);
            return response?.Entity;
        }

        private bool IsValidationComplete(Entity provider)
        {
            using (var svc = new PpmsContext(_service))
            {
                //Retrieve the Validation History Log types
                var validationHistoryLogs = svc.ppms_historylogSet
                            .Where(hl => hl.ppms_providerid.Id == provider.Id && (
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Contact ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Leie ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.NppesNpi ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Sam ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.CareSite)) 
                            .OrderByDescending(h => h.CreatedOn);

                //Add to list and check to make sure all the needed Validation Types are present

                //check the validationHistoryLogs for null / count
                if (validationHistoryLogs != null) { 
                    var validationHistoryLogsList = validationHistoryLogs.ToList();
                    var recordSource = provider.GetAttributeValue<OptionSetValue>("ppms_recordsource");
                    var providerType = provider.GetAttributeValue<OptionSetValue>("ppms_providertype");

                    // If record source not set, treat as user input (Provider Agreement)
                    if (recordSource == null) recordSource = new OptionSetValue((int)Account_ppms_RecordSource.ProviderAgreement);

                    Log("Source: {0}", recordSource.Value);
                    switch (recordSource.Value)
                    {
                        case (int)Account_ppms_RecordSource.VAProvider:
                            if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Leie)) { return false; }
                            if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.NppesNpi)) { return false; }
                            break;
                        default:
                            if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Contact)) { return false; }
                            if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Leie)) { return false; }
                            if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.NppesNpi)) { return false; }
                            break;
                    }

                    Log($"Checking for Type 2 provider: {providerType.Value}");
                    if (providerType.Value == (int)ppms_ProviderType.GroupPracticeAgency)
                    {
                        if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.CareSite)) { return false; }
                    }

                    //If all validations are present return true. 
                    return true; 
                }
                //If no validation logs are found return false. 
                return false;
            }            
        }

        private IList<SetStateRequest> MarkBatchDetailAsComplete(Entity provider)
        {
            var requests = new List<SetStateRequest>();

            Log("Validations Complete");
            var relatedBatchDetail = provider.RelatedEntities[new Relationship("ppms_account_batchdetail_provider")];

            if (relatedBatchDetail != null)
            {
                // Get batch detail information
                var batchDetail = relatedBatchDetail.Entities.FirstOrDefault();
                //Batch Detail record may not always be present. 
                if (batchDetail != null)
                {
                    // Set Batch Detail validation complete
                    // Set validation complete date on batch
                    var detail = new ppms_batchdetail()
                    {
                        Id = batchDetail.Id,
                        ppms_validationcompletedate = DateTime.Now
                    };
                    _service.Update(detail);
                    Log("Set batch detail validation complete date.");

                    // Check if batch is finished with validations
                    // Get batch record
                    var batchRef = batchDetail.GetAttributeValue<EntityReference>("ppms_batch");
                    if (batchRef != null)
                    {
                        // Check if batch validation is complete
                        if (IsBatchComplete(batchRef))
                        {
                            //Set Batch State/Status
                            var batchState = (int)PpmsHelper.AccountState.Active;
                            var batchStatuscode = (int)PpmsHelper.Batch_StatusCode.ValidationComplete;
                            requests.Add(PpmsHelper.GetStateRequest(batchRef, batchState, batchStatuscode));

                            // Set validation complete date on batch
                            var batch = new ppms_batch()
                            {
                                Id = batchRef.Id,
                                ppms_validationcompletedate = DateTime.Now
                            };
                            _service.Update(batch);
                            Log("Set batch validation complete date.");
                        }
                    }
                }
                else
                {
                    Log("Batch detail not found");
                }
            }

            return requests;
        }

        /// <summary>
        /// Determines if the batch detail records associated with a batch have all been validated
        /// </summary>
        /// <param name="batchRef"></param>
        /// <returns>true, if all associated batch detail </returns>
        private bool IsBatchComplete(EntityReference batchRef)
        {
            using (var context = new PpmsContext(_service))
            {
                // Calls to Any or Count do not work in plugin, must convert to list
                var query = context.ppms_batchdetailSet.Where(d => d.ppms_batchdetailId == batchRef.Id 
                    && d.StatusCode.Value == (int)ppms_batch_StatusCode.Processing
                    && d.ppms_validationcompletedate == null
                );
                if (query != null)
                {
                    var results = query.ToList();
                    return results.Count == 0;
                }
            }

            return false;
        }

        private void Log(string message, params object[] args)
        {
            _tracingService.Trace(message, args);
        }
    }
}
